<?php

use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Log;
use App\Models\System\SystemRoute;
use Livewire\Volt\Volt;
use Mpdf\Mpdf;

// ✅ Generate use statements dynamically
if (Schema::hasTable('system_routes')) {
    $controllers = SystemRoute::whereNotNull('controller')->pluck('controller')->unique();

    foreach ($controllers as $fullControllerClass) {
        try {
            $class = str_replace('\\\\', '\\', $fullControllerClass);
            if (class_exists($class)) {
                // تسجيل alias للكلاس (مثلاً LocaleController → App\Http\Controllers\LocaleController)
                class_alias($class, class_basename($class));
            }
        } catch (\Throwable $e) {
            Log::warning("Failed alias for: $fullControllerClass");
        }
    }

    // ✅ Start dynamic routing
    $routes = SystemRoute::where('is_active', true)->get();

    foreach ($routes as $route) {
        $method     = strtolower($route->method ?? 'get');
        $middleware = is_array($route->middleware) ? $route->middleware : [];
        $name       = $route->name;
        $url        = $route->url;
        $prefix      = $route->Prefix;
        $type       = strtolower($route->main_type ?? 'route');
        $controller = str_replace('\\\\', '\\', $route->controller);
        $controllerClass = $route->controller_class;

        $constraints = is_array($route->constraints) ? $route->constraints : json_decode($route->constraints, true);
        $viewJson   = is_array($route->view) ? $route->view : json_decode($route->view, true);
        $viewName   = $viewJson['name'] ?? null;
        $execution  = strtolower($viewJson['type'] ?? 'blade');

        // ✅ Volt
        if ($type === 'volt' && $viewName) {
            Volt::route($url, $viewName)
                ->middleware($middleware)
                ->name($name);
            continue;
        }

        // ✅ System View
        if ($type === 'system' && $execution === 'blade' && $viewName) {
            $routeInstance = Route::$method($url, fn() => view($viewName))
                ->name($name)
                ->middleware($middleware);

            // ✅ Apply constraints if available
            if (!empty($constraints)) {
                foreach ($constraints as $param => $pattern) {
                    $routeInstance->where($param, $pattern);
                }
            }
            continue;
        }

        // ✅ PDF View
        if ($type === 'system' && $execution === 'pdf' && $viewName) {
            $routeInstance = Route::get($url, function () use ($viewName) {
                $mpdf = new Mpdf([
                    'mode' => 'utf-8',
                    'default_font' => 'amiri',
                    'format' => 'A4',
                    'orientation' => 'P',
                    'margin_top' => 10,
                    'margin_bottom' => 10,
                    'margin_left' => 10,
                    'margin_right' => 10,
                ]);
                $html = view($viewName)->render();
                $mpdf->WriteHTML($html);
                return response($mpdf->Output('document.pdf', 'I'), 200)->header('Content-Type', 'application/pdf');
            })
                ->name($name)
                ->middleware($middleware);

            if (!empty($constraints)) {
                foreach ($constraints as $param => $pattern) {
                    $routeInstance->where($param, $pattern);
                }
            }
            continue;
        }

        // ✅ Controller Route
        if (!empty($controller)) {
            try {
                if (!empty($controllerClass)) {
                    $callable = eval("return {$controllerClass};");

                    $routeInstance = Route::$method($url, $callable)
                        ->name($name)
                        ->middleware($middleware)
                        ->prefix($prefix ?? '');

                    if (!empty($constraints)) {
                        foreach ($constraints as $param => $pattern) {
                            $routeInstance->where($param, $pattern);
                        }
                    }
                } else {
                    $routeInstance = Route::$method($url, $controller)
                        ->name($name)
                        ->middleware($middleware)
                        ->prefix($prefix ?? '');

                    if (!empty($constraints)) {
                        foreach ($constraints as $param => $pattern) {
                            $routeInstance->where($param, $pattern);
                        }
                    }
                }

                continue;
            } catch (\Throwable $e) {
                Log::error("Failed route bind for [$name]: " . $e->getMessage());
            }
        }

        // ❌ Fallback
        Route::$method($url, fn() => abort(404, 'Not implemented'))
            ->name($name)
            ->middleware($middleware);
    }
}else {
    // 🌐 اعتراض كل الـ URLs وإظهار صفحة backsoon
    Route::any('{any}', function () {
        return response()->view('backsoon', [], 503);
    })->where('any', '.*'); // ← يلتقط أي مسار URL
    return;
}
